#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

/**
 * Build output file name by inserting '_mutated' before the extension.
 * If no extension is found, '_mutated' is appended at the end.
 */
void build_output_name(const char *input, char *output, int iter) {
    const char *dot = strrchr(input, '.');
    if (dot) {
        size_t base_len = dot - input;
        strncpy(output, input, base_len);
        output[base_len] = '\0';
        sprintf(output + base_len, "_mutated_%d%s", iter, dot);
    } else {
        sprintf(output, "%s_mutated_%d", input, iter);
    }
}

/**
 * Perform m random mutations on a DNA sequence with TS:TV control.
 */
void mutate_dna(char *dna, int length, int m, double ts_tv_ratio) {
    char positions[m];
    int i, pos, j, count;

    for (i = 0; i < m; i++) {
        /** choose a unique position */
        do {
            count = 0;
            pos = rand() % length;
            for (j = 0; j < i; j++) {
                if (pos == positions[j]) count++;
            }
        } while (count != 0);

        positions[i] = pos;

        char current = dna[pos];
        char new_base;

        /** calculate transition probability */
        double p_ts = ts_tv_ratio / (ts_tv_ratio + 1.0);
        double r = (double)rand() / RAND_MAX;

        if ((current == 'A' || current == 'G') && r < p_ts) {
            /** Transition purine -> purine */
            new_base = (current == 'A') ? 'G' : 'A';
        } else if ((current == 'C' || current == 'T') && r < p_ts) {
            /** Transition pyrimidine -> pyrimidine */
            new_base = (current == 'C') ? 'T' : 'C';
        } else {
            /** Transversion: choose randomly among the two valid options */
            do {
                switch (current) {
                    case 'A': new_base = (rand() % 2) ? 'C' : 'T'; break;
                    case 'G': new_base = (rand() % 2) ? 'C' : 'T'; break;
                    case 'C': new_base = (rand() % 2) ? 'A' : 'G'; break;
                    case 'T': new_base = (rand() % 2) ? 'A' : 'G'; break;
                    default: new_base = current; /** fallback, should not occur */
                }
            } while (new_base == current);
        }

        dna[pos] = new_base;
    }
}

int main(int argc, char *argv[]) {
    if (argc != 5) {
        fprintf(stderr, "Usage: %s <dna_file> <num_mutations> <iteration> <ts_tv_ratio>\n", argv[0]);
        return 1;
    }

    srand(time(NULL));

    const char *input_name = argv[1];
    int m = atoi(argv[2]);
    int i = atoi(argv[3]);
    double ts_tv_ratio = atof(argv[4]);

    /** Open input file */
    FILE *fin = fopen(input_name, "rb");
    if (!fin) {
        perror("Error opening input file");
        return 1;
    }

    /** Get file size */
    fseek(fin, 0, SEEK_END);
    long size = ftell(fin);
    rewind(fin);

    /** Read sequence */
    char *dna = malloc(size + 1);
    if (!dna) {
        fprintf(stderr, "Memory allocation failed\n");
        fclose(fin);
        return 1;
    }
    fread(dna, 1, size, fin);
    fclose(fin);
    dna[size] = '\0';

    /** Apply mutations */
    mutate_dna(dna, size, m, ts_tv_ratio);

    /** Build output file name */
    char output_name[1024];
    build_output_name(input_name, output_name, i);

    /** Save mutated sequence */
    FILE *fout = fopen(output_name, "wb");
    if (!fout) {
        perror("Error creating output file");
        free(dna);
        return 1;
    }
    fwrite(dna, 1, size, fout);
    fclose(fout);

    printf("Mutated DNA saved into '%s'\n", output_name);

    free(dna);
    return 0;
}